import Header from "@/components/frontend/layout/Header";
import Footer from "@/components/frontend/sections/Footer";
import { Calendar, Search, Trophy, CheckCircle, Clock, LayoutGrid, List } from "lucide-react";
import React, { useState, useMemo } from "react";

// Define the interface for fixture data
interface FixtureTeam {
  name: string;
  logo: string;
}

interface FixtureData {
  id: number;
  date: string;
  time: string;
  teamA: FixtureTeam;
  teamB: FixtureTeam;
  venue: string;
  status: string;
  result: string;
}

interface FixturesProps {
  fixtures: FixtureData[];
}

const statusBadge = (status: string) => {
  if (status === "live") return (
    <span className="px-3 py-1 rounded-full bg-red-600 text-white text-xs font-bold animate-pulse shadow-lg">
      LIVE
    </span>
  );
  if (status === "completed") return (
    <span className="px-3 py-1 rounded-full bg-green-600 text-white text-xs font-bold flex items-center gap-1 shadow-lg">
      <CheckCircle className="h-3 w-3" />COMPLETED
    </span>
  );
  return (
    <span className="px-3 py-1 rounded-full bg-gray-500 text-white text-xs font-bold flex items-center gap-1 shadow-lg">
      <Clock className="h-3 w-3" />UPCOMING
    </span>
  );
};

const Fixtures = ({ fixtures }: FixturesProps) => {
  const [search, setSearch] = useState("");
  const [selectedDate, setSelectedDate] = useState("all");

  // Group fixtures by date
  const fixturesByDate = useMemo(() => {
    const filtered = fixtures.filter(f => {
      const q = search.toLowerCase();
      return (
        f.teamA.name.toLowerCase().includes(q) ||
        f.teamB.name.toLowerCase().includes(q) ||
        f.date.includes(q)
      );
    });

    const grouped = filtered.reduce((acc: { [key: string]: FixtureData[] }, fixture) => {
      const date = fixture.date;
      if (!acc[date]) {
        acc[date] = [];
      }
      acc[date].push(fixture);
      return acc;
    }, {});

    // Sort dates
    const sortedEntries = Object.entries(grouped).sort(([a], [b]) => 
      new Date(a).getTime() - new Date(b).getTime()
    );

    return sortedEntries;
  }, [fixtures, search]);

  const formatDate = (dateString: string) => {
    const date = new Date(dateString);
    const day = date.getDate();
    const month = date.toLocaleDateString('en-US', { month: 'short' }).toUpperCase();
    const year = date.getFullYear();
    
    const getDaySuffix = (day: number) => {
      if (day >= 11 && day <= 13) return 'TH';
      switch (day % 10) {
        case 1: return 'ST';
        case 2: return 'ND';
        case 3: return 'RD';
        default: return 'TH';
      }
    };

    return `${day}${getDaySuffix(day)} ${month} ${year}`;
  };

  const formatTime = (timeString: string) => {
    const [hours, minutes] = timeString.split(':');
    const hour = parseInt(hours);
    const ampm = hour >= 12 ? 'PM' : 'AM';
    const displayHour = hour === 0 ? 12 : hour > 12 ? hour - 12 : hour;
    return `${displayHour}:${minutes} ${ampm}`;
  };

  return (
    <div className="min-h-screen bg-gray-900">
      <Header />
      
      {/* Tournament Hero Section */}
      <section className="relative min-h-[400px] flex items-center justify-center overflow-hidden">
        {/* Stadium Background */}
        <div 
          className="absolute inset-0 bg-cover bg-center bg-no-repeat"
          style={{
            backgroundImage: `url('data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 1920 1080"><defs><radialGradient id="light1" cx="20%" cy="10%"><stop offset="0%" stop-color="%23ffffff" stop-opacity="0.8"/><stop offset="100%" stop-color="%23ffffff" stop-opacity="0"/></radialGradient><radialGradient id="light2" cx="80%" cy="10%"><stop offset="0%" stop-color="%23ffffff" stop-opacity="0.8"/><stop offset="100%" stop-color="%23ffffff" stop-opacity="0"/></radialGradient><radialGradient id="light3" cx="50%" cy="20%"><stop offset="0%" stop-color="%23ffffff" stop-opacity="0.6"/><stop offset="100%" stop-color="%23ffffff" stop-opacity="0"/></radialGradient></defs><rect width="100%" height="100%" fill="%23001829"/><ellipse cx="384" cy="108" rx="200" ry="100" fill="url(%23light1)"/><ellipse cx="1536" cy="108" rx="200" ry="100" fill="url(%23light2)"/><ellipse cx="960" cy="216" rx="300" ry="150" fill="url(%23light3)"/></svg>')`
          }}
        />
        
        {/* Stadium Lights Effects */}
        <div className="absolute inset-0">
          <div className="absolute top-0 left-1/4 w-32 h-32 bg-white opacity-20 rounded-full blur-3xl animate-pulse"></div>
          <div className="absolute top-0 right-1/4 w-32 h-32 bg-white opacity-20 rounded-full blur-3xl animate-pulse" style={{ animationDelay: '1s' }}></div>
          <div className="absolute top-0 left-1/2 transform -translate-x-1/2 w-40 h-40 bg-white opacity-15 rounded-full blur-3xl animate-pulse" style={{ animationDelay: '2s' }}></div>
        </div>

        {/* Blue Gradient Overlay */}
        <div className="absolute inset-0 bg-gradient-to-b from-blue-900/80 via-blue-800/90 to-gray-900/95"></div>
        
        {/* Content */}
        <div className="relative z-10 text-center text-white px-4 max-w-6xl mx-auto">
          {/* Tournament Title */}
          <h1 className="text-3xl md:text-5xl lg:text-6xl font-extrabold mb-8 leading-tight drop-shadow-2xl">
            THE LATE EV. GERISHOM MILIMU<br />
            <span className="text-green-700">CUP FOOTBALL TOURNAMENT</span><br />
            <span className="text-2xl md:text-3xl lg:text-4xl text-gray-300">3RD EDITION</span>
          </h1>
          
          {/* Fixtures Banner */}
          <div className="relative inline-block">
            {/* Decorative Elements */}
            <div className="absolute -left-16 top-1/2 transform -translate-y-1/2 w-12 h-12 bg-red-600 rotate-45 opacity-80 hidden md:block"></div>
            <div className="absolute -right-16 top-1/2 transform -translate-y-1/2 w-12 h-12 bg-red-600 rotate-45 opacity-80 hidden md:block"></div>
            
            {/* Main Banner */}
            <div className="bg-gradient-to-r from-gray-700 via-gray-600 to-gray-700 px-12 py-6 rounded-lg shadow-2xl border-4 border-red-600">
              <h2 className="text-3xl md:text-4xl font-bold text-white tracking-wider drop-shadow-lg">
                FIXTURES
              </h2>
            </div>
          </div>
        </div>
      </section>

      {/* Search Section */}
      <section className="bg-gray-800 py-6 px-4">
        <div className="max-w-6xl mx-auto">
          <div className="flex flex-col md:flex-row gap-4 items-center justify-center">
            <div className="flex items-center gap-2">
              <Search className="h-5 w-5 text-gray-400" />
              <input
                type="text"
                placeholder="Search teams..."
                className="w-full md:w-80 px-4 py-3 rounded-lg border border-gray-600 bg-gray-700 text-white focus:outline-none focus:ring-2 focus:ring-red-500 focus:border-transparent"
                value={search}
                onChange={e => setSearch(e.target.value)}
              />
            </div>
          </div>
        </div>
      </section>

      {/* Fixtures Content */}
      <section className="py-8 px-4 bg-gray-900 min-h-screen">
        <div className="max-w-7xl mx-auto">
          {fixturesByDate.length === 0 ? (
            <div className="text-center text-gray-400 py-16">
              <Calendar className="mx-auto h-16 w-16 mb-4 opacity-50" />
              <h3 className="text-xl font-semibold mb-2">No fixtures found</h3>
              <p>Check back later for upcoming matches.</p>
            </div>
          ) : (
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
              {fixturesByDate.map(([date, dayFixtures], index) => (
                <div key={date} className="space-y-6">
                  {/* Date Header */}
                  <div className="text-center">
                    <h3 className="text-2xl md:text-3xl font-bold text-white bg-gradient-to-r from-red-600 to-red-700 px-6 py-3 rounded-lg shadow-lg inline-block">
                      {formatDate(date)}
                    </h3>
                  </div>
                  
                  {/* Matches for this date */}
                  <div className="space-y-4">
                    {dayFixtures.map((fixture) => (
                      <div key={fixture.id} className="bg-gray-800 rounded-lg shadow-xl overflow-hidden border border-gray-700 hover:border-red-500 transition-all duration-300">
                        <div className="flex items-center">
                          {/* Time Section */}
                          <div className="bg-red-600 text-white px-4 py-6 text-center min-w-[100px]">
                            <div className="text-lg font-bold">
                              {formatTime(fixture.time)}
                            </div>
                          </div>
                          
                          {/* Teams Section */}
                          <div className="flex-1 p-4">
                            <div className="flex items-center justify-between">
                              {/* Team A */}
                              <div className="flex items-center space-x-3 flex-1">
                                <img 
                                  src={fixture.teamA.logo} 
                                  alt={fixture.teamA.name}
                                  className="w-8 h-8 rounded-full object-cover border-2 border-gray-600"
                                />
                                <span className="text-white font-semibold text-sm md:text-base truncate">
                                  {fixture.teamA.name}
                                </span>
                              </div>
                              
                              {/* VS */}
                              <div className="mx-4">
                                <span className="text-gray-400 font-bold text-lg">VS</span>
                              </div>
                              
                              {/* Team B */}
                              <div className="flex items-center space-x-3 flex-1 justify-end">
                                <span className="text-white font-semibold text-sm md:text-base truncate">
                                  {fixture.teamB.name}
                                </span>
                                <img 
                                  src={fixture.teamB.logo} 
                                  alt={fixture.teamB.name}
                                  className="w-8 h-8 rounded-full object-cover border-2 border-gray-600"
                                />
                              </div>
                            </div>
                            
                            {/* Match Info */}
                            <div className="mt-3 flex items-center justify-between text-sm">
                              <span className="text-gray-400">{fixture.venue}</span>
                              <div className="flex items-center gap-3">
                                <span className="text-white font-bold">{fixture.result}</span>
                                {statusBadge(fixture.status)}
                              </div>
                            </div>
                          </div>
                        </div>
                      </div>
                    ))}
                  </div>
                </div>
              ))}
            </div>
          )}
        </div>
      </section>

      {/* Tournament Footer */}
      <section className="bg-gradient-to-r from-green-800 to-green-900 py-8 px-4 text-center text-white">
        <div className="max-w-4xl mx-auto">
          <h2 className="text-3xl font-bold mb-6">IN GOD WE TRUST</h2>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6 text-sm">
            <div className="bg-black/20 p-4 rounded-lg">
              <h3 className="font-bold mb-2 text-green-700">TOURNAMENT RULES</h3>
              <ul className="space-y-1 text-left">
                <li>• All teams to arrive 1 hour before kickoff time</li>
                <li>• Penalty of Ksh 500 lateness rule to apply</li>
                <li>• 25th Christmas, and 31st all teams, fans to take a break</li>
              </ul>
            </div>
            <div className="bg-black/20 p-4 rounded-lg">
              <h3 className="font-bold mb-2 text-green-700">MATCH DETAILS</h3>
              <ul className="space-y-1 text-left">
                <li>• Entertainment and teams pre-match to take 45 minutes</li>
                <li>• No cash award - all teams to be awarded by cheque</li>
                <li>• Award ceremony: Emcee/DJ from 5:45 PM to 6:30 PM</li>
              </ul>
            </div>
          </div>
        </div>
      </section>

      <Footer />
    </div>
  );
};

export default Fixtures; 